{
    This file is part of the ChebLib library,
    Copyright (c) 2004 by Anton Rzheshevski (chebmaster@mail.ru),
      and contains classes for working with file lists 
      and directory trees.
      
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **********************************************************************}
{$mode delphi}
 
unit cl_filelist;

interface

uses SysUtils, Classes;

type
  TFileList = class(TStringList)
  private
    SortedByDate: boolean;
    function _time(ind: integer): TDateTime;
    function _size(ind: integer): integer;
    function _attr(ind: integer): integer;
    function _sorted: boolean;
    function _msdostimestamp(index: integer): integer;
    function _names(index: integer): string;
  public
    constructor Create(pathmask: string); overload;
    constructor Create(pathmask: array of string); overload;
    procedure SortByName;
    procedure Sort; OVERRIDE;
    procedure SortByDate; //default
    destructor Destroy; Override;
    Property Time[index: integer]: TDateTime read _time;
    property MSDOSTimeStamp[index: integer]: integer read _msdostimestamp;
    Property Size[index: integer]: integer read _size;
    Property Attributes[index: integer]: integer read _attr;
    Property Sorted: boolean read _sorted;
    Property Names[index: integer]: string read _names;
  end;

  TDirectoryList = class; //forward declaration

  TDirectory = class
    Constructor Create(path: string);
    Destructor Destroy; Override;
  private
    _files: TFileList;
    _dirs: TDirectoryList;
    _name, _path: string;
  public
    property Files: TFileList read _files;
    property Directories: TDirectoryList read _dirs;
    property Name: string read _name;
    property Path: string read _path;
  end;

  TDirectoryList = class(TStringList)
  private
    function _d(ind: integer): TDirectory;
  public
    constructor Create (path: string);
    destructor Destroy; Override;
    property Dir[index: integer]:TDirectory read _d;
  end;

implementation

type
  TFileRecord = class
  public
    Time: TDateTime;
    timestamp,
    Size,
    Attributes: Integer;
    Fullname: string;
  end;

{ TFileList }

function TFileList._attr(ind: integer): integer;
begin
  Result:=TFileRecord(Objects[ind]).Attributes;
end;

function TFileList._size(ind: integer): integer;
begin
  Result:=TFileRecord(Objects[ind]).Size;
end;

function TFileList._sorted: boolean;
begin
  Result:=not SortedByDate;
end;

function TFileList._time(ind: integer): TDateTime;
begin
  Result:=TFileRecord(Objects[ind]).time;
end;

function TFileList._msdostimestamp(index: integer): integer;
begin
  Result:=TFileRecord(Objects[index]).timestamp;
end;

constructor TFileList.Create(pathmask: string);
var
  SearchRec: TSearchRec;
  r: TFileRecord;
begin
  inherited Create;
  If FindFirst(pathmask, faAnyFile, SearchRec) = 0
  then Repeat
    r:=TFileRecord.Create;

                                    //crashes in Linux. (?!?..)
    r.Time:={$ifdef unix} 0; {$else}FileDateToDateTime(SearchRec.Time);{$endif}

    r.TimeStamp:=SearchRec.Time;
    r.Size:=SearchRec.Size;
    r.Attributes:=SearchRec.Attr;
    r.Fullname:=ExtractFilePath(pathmask) + SearchRec.Name;
    AddObject(intToHex(SearchRec.Time,8) + SearchRec.Name, r);
    r:=nil;
  Until FindNext(SearchRec)<>0;
  SortByDate;
end;

constructor TFileList.Create(pathmask: array of string);
var
  SearchRec: TSearchRec;
  r: TFileRecord;
  i: integer;
begin
  inherited Create;
  For i:=0 to Length(pathmask) - 1 do begin
    If FindFirst(pathmask[i], faAnyFile, SearchRec) = 0
    then Repeat
      r:=TFileRecord.Create;

                                      //crashes in Linux. (?!?..)
      r.Time:={$ifdef unix} 0; {$else}FileDateToDateTime(SearchRec.Time);{$endif}

      r.TimeStamp:=SearchRec.Time;
      r.Size:=SearchRec.Size;
      r.Attributes:=SearchRec.Attr;
      r.Fullname:=ExtractFilePath(pathmask[i]) + SearchRec.Name;
      AddObject(intToHex(SearchRec.Time,8) + SearchRec.Name, r);
      r:=nil;
    Until FindNext(SearchRec)<>0;
  end;
  SortByDate;
end;

destructor TFileList.Destroy;
var j: integer;
begin
  For j:=0 to Count-1 do begin
    Objects[j].Free;
    Objects[j]:=nil;
  end;
  inherited;
end;

procedure TFileList.Sort;
begin
  SortByName;
end;

procedure TFileList.SortByDate;
var j: integer;
begin
  For j:=0 to Count-1 do
    Strings[j]:=IntToHex(MSDOSTimeStamp[j], 8) + Strings[j];
  inherited Sort;
  For j:=0 to Count-1 do
    Strings[j]:=Copy(Strings[j], 9, Length(Strings[j]) - 8);
  SortedByDate:=True;
end;

procedure TFileList.SortByName;
begin
  SortedByDate:=false;
  inherited;
end;


function TFileList._names(index: integer): string;
begin
  Result:=TFileRecord(Objects[index]).Fullname;
end;

{ TDirectory }

constructor TDirectory.Create(path: string);
begin
  inherited Create;
  _files:=TFileList.Create(path+ '*.*');
  _path:=path;
  //_name:=
  _dirs:=TDirectoryList.Create(path);
end;

destructor TDirectory.Destroy;
begin
  _files.Free;
  _dirs.Free;
  inherited;
end;

{ TDirectoryList }

function TDirectoryList._d(ind: integer): TDirectory;
begin
  Result:=TDirectory(Objects[ind]);
end;

constructor TDirectoryList.Create(path: string);
var
  SearchRec: TSearchRec;
  d: TDirectory;
begin
  inherited Create;
  If FindFirst(path+'*.*', faDirectory, SearchRec) = 0
  then Repeat
    if (SearchRec.Name <> '.') and (SearchRec.Name <> '..')
    then begin
      d:=TDirectory.Create(path + SearchRec.Name + '\');
      AddObject(d.Path, d);
    end;
  Until FindNext(SearchRec)<>0;
  Sort;
end;

destructor TDirectoryList.Destroy;
var j: integer;
begin
  For j:=0 to Count-1 do begin
    Objects[j].Free;
    Objects[j]:=nil;
  end;
  inherited;
end;

end.
